﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins.Agreement
{
    public class AgreementUpdate : IPlugin
    {
        private ITracingService _tracingService;
        private const string PluginName = "AgreementUpdate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity agreement = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (agreement.LogicalName != "ppms_provideragreement")
                {
                    _tracingService.Trace("Entity type does not match! [{0}]", agreement.LogicalName);
                    return;
                }

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                IOrganizationService service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Get related entities to deactivate
                    _tracingService.Trace("Retrieve Agreement and related entities");
                    Entity entity = GetAgreement(service, agreement.Id);
                    if (entity == null)
                    {
                        _tracingService.Trace("Failed to retrieve agreement and related entities");
                        return;
                    }
                    _tracingService.Trace("Agreement Retrieved");

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    _tracingService.Trace("Check status");
                    var isActive = IsActive(entity);
                    int state = 0;
                    int statuscode = 0;

                    if (isActive)
                    {
                        _tracingService.Trace("Agreement in Active Status");
                        state = (int)ppms_providerserviceState.Active;
                        statuscode = (int)ppms_providerservice_StatusCode.Active;
                        // If agreement is not Active/Active, provider service status should be Pending
                        if (!PpmsHelper.DoesStatusMatch(entity, state, statuscode))
                            statuscode = (int)ppms_providerservice_StatusCode.Pending;
                    }
                    else
                    {
                        _tracingService.Trace("Agreement in Inactive Status");
                        state = (int)ppms_providerserviceState.Inactive;
                        statuscode = (int)ppms_providerservice_StatusCode.Inactive;
                    }

                    _tracingService.Trace("New provider service - status: {0}, state: {1}", state, statuscode);

                    // provider service
                    requests.AddRange(PpmsHelper.GetEntityStateRequests(entity, "ppms_provideragreement_ppms_providerservice_ProviderAgreement", state, statuscode, true));

                    if (requests != null)
                    {
                        _tracingService.Trace("Executing Set State Requests [{0}]", requests.Count);
                        foreach (var request in requests)
                        {
                            service.Execute(request);
                        }
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private Entity GetAgreement(IOrganizationService service, Guid agreementId)
        {
            var request = new RetrieveRequest
            {
                Target = new EntityReference("ppms_provideragreement", agreementId),
                ColumnSet = new ColumnSet(new string[] { "ppms_provideragreementid", "ppms_agreementid", "statecode", "statuscode" }),
                RelatedEntitiesQuery = new RelationshipQueryCollection
                {
                    {
                        new Relationship("ppms_provideragreement_ppms_providerservice_ProviderAgreement"),
                        new QueryExpression("ppms_providerservice")
                        {
                            ColumnSet = new ColumnSet("ppms_providerserviceid", "ppms_name", "statecode", "statuscode")
                        }
                    }
                }
            };

            //Get response of request
            var response = (RetrieveResponse)service.Execute(request);
            if (response != null)
                return response.Entity;

            return null;
        }

        /// <summary>
        /// Get state of the specified entity
        /// </summary>
        /// <param name="entity">Entity to check</param>
        /// <returns>true if in Active state and Active status code</returns>
        private bool IsActive(Entity entity)
        {
            var statecode = entity.GetAttributeValue<OptionSetValue>("statecode");
            var statuscode = entity.GetAttributeValue<OptionSetValue>("statuscode");
            _tracingService.Trace("Agreement - State: {0}, Status: {1}", statecode?.Value, statuscode?.Value);
            return statecode != null && statecode.Value == (int)AccountState.Active ? true : false;
        }
    }
}
